<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

requireLogin();
$activePage = 'invoice';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Check permissions
$isSuperAdmin = false;
if (isset($_SESSION['user_id'])) {
    $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = (SELECT role_id FROM users WHERE id = ?)");
    $stmt->execute([$_SESSION['user_id']]);
    $isSuperAdmin = $stmt->fetchColumn();
}

$userName = $_SESSION['username'] ?? 'User';

// Get store settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
$stmt->execute();
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get invoices with payment information (handle case where table doesn't exist)
$invoices = [];
try {
    $stmt = $pdo->prepare("
        SELECT 
            i.*,
            c.name as customer_name,
            c.email as customer_email,
            c.phone as customer_phone,
            u.username as created_by_name,
            COALESCE(q.quote_number, '') as quotation_reference,
            COALESCE(SUM(p.amount), 0) as total_paid
        FROM invoices i
        LEFT JOIN customers c ON i.customer_id = c.id
        JOIN users u ON i.created_by = u.id
        LEFT JOIN quotations q ON i.quotation_id = q.id
        LEFT JOIN payments p ON i.id = p.invoice_id
        GROUP BY i.id
        ORDER BY i.created_at DESC
        LIMIT 50
    ");
    $stmt->execute();
    $invoices = $stmt->fetchAll();
} catch (PDOException $e) {
    // If payments table doesn't exist, get invoices without payment info
    try {
        $stmt = $pdo->prepare("
            SELECT 
                i.*,
                c.name as customer_name,
                c.email as customer_email,
                c.phone as customer_phone,
                u.username as created_by_name,
                COALESCE(q.quote_number, '') as quotation_reference,
                0 as total_paid
            FROM invoices i
            LEFT JOIN customers c ON i.customer_id = c.id
            JOIN users u ON i.created_by = u.id
            LEFT JOIN quotations q ON i.quotation_id = q.id
            ORDER BY i.created_at DESC
            LIMIT 50
        ");
        $stmt->execute();
        $invoices = $stmt->fetchAll();
    } catch (PDOException $e2) {
    // Table doesn't exist yet - will be created later
    $invoices = [];
    }
}

// Get customers for dropdown
$stmt = $pdo->prepare("SELECT id, name, email, phone FROM customers ORDER BY name");
$stmt->execute();
$customers = $stmt->fetchAll();

// Get products for invoice items
$stmt = $pdo->prepare("SELECT id, name, price FROM products ORDER BY name");
$stmt->execute();
$products = $stmt->fetchAll();

// Fetch quotations for dropdown
$quotations = [];
try {
    $stmt = $pdo->prepare("
        SELECT 
            q.id,
            q.quote_number,
            q.customer_id,
            c.name as customer_name,
            c.email as customer_email,
            c.phone as customer_phone
        FROM quotations q
        LEFT JOIN customers c ON q.customer_id = c.id
        WHERE q.status = 'active' OR q.status = 'converted'
        ORDER BY q.created_at DESC
    ");
    $stmt->execute();
    $quotations = $stmt->fetchAll();
} catch (PDOException $e) {
    $quotations = [];
}

// Calculate invoice statistics
$stats = [
    'total_invoices' => count($invoices),
    'paid_invoices' => count(array_filter($invoices, fn($inv) => $inv['status'] === 'paid')),
    'pending_invoices' => count(array_filter($invoices, fn($inv) => $inv['status'] === 'pending')),
    'overdue_invoices' => count(array_filter($invoices, fn($inv) => $inv['status'] === 'overdue')),
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice Management - <?php echo htmlspecialchars($settings['store_name'] ?? 'POS System'); ?></title>
    
    <!-- Favicon -->
    <?php echo getFaviconLink($pdo); ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Custom CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    
    <style>
        :root {
            --primary-color: #2563eb;
            --primary-dark: #1d4ed8;
            --secondary-color: #64748b;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #06b6d4;
            --light-bg: #f8fafc;
            --border-color: #e2e8f0;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        }

        * {
            font-family: 'Inter', sans-serif;
        }

        body {
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%);
            color: var(--text-primary);
            min-height: 100vh;
        }

        .main-content {
            margin-left: 280px;
            padding: 2rem;
            transition: all 0.3s ease;
        }
        
        @media (max-width: 991px) {
            .main-content {
                margin-left: 0;
                padding: 1rem;
                padding-top: 80px;
            }
        }

        /* Header Section */
        .page-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 20px;
            padding: 1rem;
            margin-bottom: 2rem;
            color: white;
            box-shadow: var(--shadow-lg);
            position: relative;
            overflow: hidden;
        }

        .page-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/><circle cx="10" cy="60" r="0.5" fill="white" opacity="0.1"/><circle cx="90" cy="40" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
            opacity: 0.3;
        }

        .page-header h1 {
            font-size: 1.75rem;
            font-weight: 700;
            margin: 0;
            position: relative;
            z-index: 1;
        }

        .page-header p {
            font-size: 0.9rem;
            opacity: 0.9;
            margin: 0.25rem 0 0 0;
            position: relative;
            z-index: 1;
        }

        .header-actions {
            position: relative;
            z-index: 1;
        }

        /* Cards */
        .card {
            border: none;
            border-radius: 16px;
            box-shadow: var(--shadow-md);
            background: white;
            transition: all 0.3s ease;
            overflow: hidden;
        }

        .card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }
        
        .card-header {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border-bottom: 1px solid var(--border-color);
            padding: 0.75rem;
            border-radius: 16px 16px 0 0 !important;
        }

        .card-header h5 {
            font-weight: 600;
            color: var(--text-primary);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1rem;
        }

        .card-body {
            padding: 0.75rem;
        }

        /* Buttons */
        .btn {
            border-radius: 12px;
            font-weight: 500;
            padding: 0.75rem 1.5rem;
            transition: all 0.3s ease;
            border: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.3);
        }

        .btn-success {
            background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%);
            color: white;
        }

        .btn-info {
            background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%);
            color: white;
        }

        .btn-outline-primary {
            border: 2px solid var(--primary-color);
            color: var(--primary-color);
            background: transparent;
        }

        .btn-outline-primary:hover {
            background: var(--primary-color);
            color: white;
            transform: translateY(-2px);
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            border-radius: 8px;
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 0.375rem;
            box-shadow: var(--shadow-md);
            transition: all 0.3s ease;
            border-left: 4px solid var(--primary-color);
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .stat-card .stat-icon {
            width: 25px;
            height: 25px;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.25rem;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
            font-size: 0.75rem;
        }

        .stat-card .stat-value {
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.0625rem;
        }

        .stat-card .stat-label {
            color: var(--text-secondary);
            font-size: 0.625rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Search Bar */
        .search-container {
            position: relative;
            max-width: 400px;
        }

        .search-container .form-control {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 0.75rem 1rem 0.75rem 3rem;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: white;
        }

        .search-container .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .search-container .search-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
            z-index: 2;
        }

        /* Table */
        .table {
            margin: 0;
            border-radius: 12px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border: none;
            padding: 0.5rem;
            font-weight: 600;
            color: var(--text-primary);
            text-transform: uppercase;
            font-size: 0.75rem;
            letter-spacing: 0.5px;
        }

        .table td {
            padding: 0.5rem;
            border: none;
            border-bottom: 1px solid var(--border-color);
            vertical-align: middle;
        }

        .table tbody tr {
            transition: all 0.3s ease;
        }

        .table tbody tr:hover {
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%);
            transform: scale(1.01);
        }

        .table tbody tr:last-child td {
            border-bottom: none;
        }

        /* Badges */
        .badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-weight: 500;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .bg-success {
            background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%) !important;
        }

        .bg-danger {
            background: linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%) !important;
        }

        .bg-warning {
            background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%) !important;
            color: white !important;
        }

        .bg-info {
            background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%) !important;
        }

        /* Action Buttons */
        .btn-group {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }

        .btn-group .btn {
            border-radius: 8px;
            padding: 0.5rem;
            min-width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 3rem 2rem;
            color: var(--text-secondary);
        }

        .empty-state i {
            font-size: 4rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        .empty-state h5 {
            font-size: 1.25rem;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        .empty-state p {
            font-size: 1rem;
            margin: 0;
        }

        /* Form Controls */
        .form-control, .form-select {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 0.75rem 1rem;
            transition: all 0.3s ease;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .form-label {
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        /* Modal Enhancements */
        .modal-content {
            border: none;
            border-radius: 20px;
            box-shadow: var(--shadow-lg);
        }

        .modal-header {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border-bottom: 1px solid var(--border-color);
            border-radius: 20px 20px 0 0;
            padding: 1.5rem;
        }

        .modal-title {
            font-weight: 600;
            color: var(--text-primary);
        }

        .modal-body {
            padding: 1.5rem;
        }

        .modal-footer {
            border-top: 1px solid var(--border-color);
            padding: 1.5rem;
            border-radius: 0 0 20px 20px;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .page-header h1 {
                font-size: 2rem;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .btn-group {
                flex-direction: column;
            }
            
            .btn-group .btn {
                width: 100%;
            }
        }

        /* Loading Animation */
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Alert Enhancements */
        .alert {
            border: none;
            border-radius: 12px;
            padding: 1rem 1.5rem;
            margin-bottom: 1rem;
            box-shadow: var(--shadow-sm);
        }

        .alert-success {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            color: #065f46;
        }

        .alert-danger {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            color: #991b1b;
        }

        .alert-warning {
            background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
            color: #92400e;
        }

        .alert-info {
            background: linear-gradient(135deg, #cffafe 0%, #a5f3fc 100%);
            color: #155e75;
        }

        /* Invoice Items Scrollable Container */
        .invoice-items-container {
            max-height: 400px;
            overflow-y: auto;
            border: 1px solid var(--border-color);
            border-radius: 12px;
            padding: 1rem;
            background: #f8fafc;
        }

        .invoice-items-container::-webkit-scrollbar {
            width: 6px;
        }

        .invoice-items-container::-webkit-scrollbar-track {
            background: #e2e8f0;
            border-radius: 3px;
        }

        .invoice-items-container::-webkit-scrollbar-thumb {
            background: var(--primary-color);
            border-radius: 3px;
        }

        .invoice-items-container::-webkit-scrollbar-thumb:hover {
            background: var(--primary-dark);
        }

        .invoice-item {
            background: white;
            border-radius: 8px;
            padding: 0.75rem;
            margin-bottom: 0.5rem;
            border: 1px solid var(--border-color);
            transition: all 0.3s ease;
        }

        .invoice-item:hover {
            box-shadow: var(--shadow-sm);
            border-color: var(--primary-color);
        }

        .invoice-item:last-child {
            margin-bottom: 0;
        }

        .invoice-items-header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
            padding: 0.75rem 1rem;
            border-radius: 12px 12px 0 0;
            margin: -1rem -1rem 1rem -1rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
    </style>
</head>
<body>
    <!-- Include Sidebar -->
    <?php include '../includes/sidebar.php'; ?>
    <?php include '../includes/topbar.php'; ?>
    
    <!-- Alert Container -->
    <div id="alertContainer" class="position-fixed top-0 end-0 p-3" style="z-index: 9999;"></div>
    
    <div class="main-content">
        <!-- Page Header -->
        <div class="page-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-receipt me-3"></i>Invoices</h1>
                    <p>Manage and track all your business invoices</p>
                </div>
                <div class="header-actions">
            <?php if (hasPermission('add_invoice') || hasPermission('view_invoice')): ?>
                    <button class="btn btn-light" data-bs-toggle="modal" data-bs-target="#addInvoiceModal">
                        <i class="fas fa-plus"></i>
                        Create Invoice
            </button>
            <?php endif; ?>
        </div>
                    </div>
                </div>
        
        <!-- Stats Cards -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-receipt"></i>
            </div>
                <div class="stat-value"><?php echo $stats['total_invoices']; ?></div>
                <div class="stat-label">Total Invoices</div>
                    </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="stat-value"><?php echo $stats['paid_invoices']; ?></div>
                <div class="stat-label">Paid Invoices</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%);">
                    <i class="fas fa-clock"></i>
                    </div>
                <div class="stat-value"><?php echo $stats['pending_invoices']; ?></div>
                <div class="stat-label">Pending Invoices</div>
                </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%);">
                    <i class="fas fa-exclamation-triangle"></i>
            </div>
                <div class="stat-value"><?php echo $stats['overdue_invoices']; ?></div>
                <div class="stat-label">Overdue Invoices</div>
            </div>
        </div>
        
        <!-- Invoices Table -->
        <div class="card">
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h5>
                        <i class="fas fa-list"></i>
                        All Invoices
                </h5>
                    <div class="search-container">
                        <i class="fas fa-search search-icon"></i>
                        <input type="text" class="form-control" id="invoiceSearchInput" placeholder="Search invoices...">
            </div>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Invoice #</th>
                                <th>Customer</th>
                                <th>Amount</th>
                                <th>Paid</th>
                                <th>Balance</th>
                                <th>Status</th>
                                <th>Issue Date</th>
                                <th>Due Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($invoices)): ?>
                                <tr>
                                    <td colspan="9">
                                        <div class="empty-state">
                                            <i class="fas fa-receipt"></i>
                                            <h5>No invoices found</h5>
                                            <p>Create your first invoice to get started!</p>
                                            <button class="btn btn-primary mt-3" data-bs-toggle="modal" data-bs-target="#addInvoiceModal">
                                                <i class="fas fa-plus me-2"></i>Create Invoice
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($invoices as $invoice): ?>
                                    <?php 
                                    $totalPaid = floatval($invoice['total_paid'] ?? 0);
                                    $balance = floatval($invoice['total_amount']) - $totalPaid;
                                    ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($invoice['invoice_number']); ?></td>
                                        <td><?php echo htmlspecialchars($invoice['customer_name'] ?? 'Walk-in Customer'); ?></td>
                                        <td>Ksh <?php echo number_format($invoice['total_amount'], 2); ?></td>
                                        <td>Ksh <?php echo number_format($totalPaid, 2); ?></td>
                                        <td>
                                            <?php if ($balance > 0): ?>
                                                <span class="text-danger">Ksh <?php echo number_format($balance, 2); ?></span>
                                            <?php else: ?>
                                                <span class="text-success">Ksh 0.00</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge <?php 
                                                echo match($invoice['status']) {
                                                    'paid' => 'bg-success',
                                                    'pending' => 'bg-warning',
                                                    'overdue' => 'bg-danger',
                                                    'cancelled' => 'bg-secondary',
                                                    default => 'bg-info'
                                                };
                                            ?>">
                                                <?php echo ucfirst($invoice['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('M j, Y', strtotime($invoice['issue_date'])); ?></td>
                                        <td><?php echo date('M j, Y', strtotime($invoice['due_date'])); ?></td>
                                        <td>
                                            <div class="btn-group">
                                                <?php if (hasPermission('view_invoice')): ?>
                                                <button class="btn btn-outline-primary view-invoice" title="View Invoice" data-id="<?php echo $invoice['id']; ?>" data-invoice-number="<?php echo htmlspecialchars($invoice['invoice_number']); ?>">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if ($invoice['status'] !== 'paid' && hasPermission('edit_invoice')): ?>
                                                <button class="btn btn-outline-info edit-invoice" title="Edit" data-id="<?php echo $invoice['id']; ?>" data-invoice-number="<?php echo htmlspecialchars($invoice['invoice_number']); ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if ($balance > 0 && hasPermission('pay_invoice')): ?>
                                                <button class="btn btn-outline-success pay-invoice" title="Pay Invoice" data-id="<?php echo $invoice['id']; ?>" data-invoice-number="<?php echo htmlspecialchars($invoice['invoice_number']); ?>" data-amount="<?php echo $invoice['total_amount']; ?>">
                                                    <i class="fas fa-credit-card"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('print_invoice')): ?>
                                                <button class="btn btn-outline-secondary print-invoice" title="Print Invoice" data-id="<?php echo $invoice['id']; ?>" data-invoice-number="<?php echo htmlspecialchars($invoice['invoice_number']); ?>">
                                                    <i class="fas fa-print"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('send_invoice_reminder')): ?>
                                                <button class="btn btn-outline-warning send-reminder" title="Send WhatsApp Reminder" data-id="<?php echo $invoice['id']; ?>" data-invoice-number="<?php echo htmlspecialchars($invoice['invoice_number']); ?>">
                                                    <i class="fas fa-whatsapp"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('delete_invoice')): ?>
                                                <button class="btn btn-outline-danger delete-invoice" title="Delete" data-id="<?php echo $invoice['id']; ?>" data-invoice-number="<?php echo htmlspecialchars($invoice['invoice_number']); ?>">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Invoice Modal -->
    <div class="modal fade" id="addInvoiceModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-receipt me-2"></i>
                        Create New Invoice
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="addInvoiceForm">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-12">
                                <div class="mb-3">
                                    <label for="quotation_id" class="form-label">
                                        <i class="fas fa-file-invoice me-2"></i>Select from Quotation (Optional)
                                    </label>
                                    <select class="form-select" id="quotation_id" name="quotation_id">
                                        <option value="">Create new invoice</option>
                                        <?php if (empty($quotations)): ?>
                                            <option value="" disabled>No quotations available</option>
                                        <?php else: ?>
                                            <?php foreach ($quotations as $quotation): ?>
                                                <option value="<?php echo $quotation['id']; ?>"
                                                    data-customer-id="<?php echo $quotation['customer_id'] ?? ''; ?>"
                                                    data-customer-name="<?php echo htmlspecialchars($quotation['customer_name'] ?? ''); ?>"
                                                    data-customer-email="<?php echo htmlspecialchars($quotation['customer_email'] ?? ''); ?>"
                                                    data-customer-phone="<?php echo htmlspecialchars($quotation['customer_phone'] ?? ''); ?>">
                                                    <?php echo htmlspecialchars($quotation['quote_number']); ?> -
                                                    <?php echo htmlspecialchars($quotation['customer_name'] ?? ''); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <div class="form-text">
                                        <i class="fas fa-info-circle me-1"></i>
                                        Select a quotation to automatically populate customer details.
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="customer_id" class="form-label">Customer</label>
                                    <select class="form-select" id="customer_id" name="customer_id" required>
                                        <option value="">Select Customer</option>
                                        <?php foreach ($customers as $customer): ?>
                                            <option value="<?php echo $customer['id']; ?>" 
                                                    data-email="<?php echo htmlspecialchars($customer['email'] ?? ''); ?>"
                                                    data-phone="<?php echo htmlspecialchars($customer['phone'] ?? ''); ?>">
                                                <?php echo htmlspecialchars($customer['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="issue_date" class="form-label">Issue Date</label>
                                    <input type="date" class="form-control" id="issue_date" name="issue_date" required>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="mb-3">
                                    <label for="due_date" class="form-label">Due Date</label>
                                    <input type="date" class="form-control" id="due_date" name="due_date" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="customer_email" class="form-label">Customer Email</label>
                                    <input type="email" class="form-control" id="customer_email" name="customer_email">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="payment_terms" class="form-label">Payment Terms</label>
                                    <select class="form-select" id="payment_terms" name="payment_terms">
                                        <option value="net_30">Net 30 Days</option>
                                        <option value="net_15">Net 15 Days</option>
                                        <option value="net_7">Net 7 Days</option>
                                        <option value="due_on_receipt">Due on Receipt</option>
                                        <option value="cash_on_delivery">Cash on Delivery</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="notes" class="form-label">Notes</label>
                            <textarea class="form-control" id="notes" name="notes" rows="2" placeholder="Additional notes or terms..."></textarea>
                        </div>
                        
                        <!-- Invoice Items -->
                        <div class="invoice-items-container">
                            <div class="invoice-items-header">
                                <i class="fas fa-list"></i>
                                Invoice Items <span id="itemCount" class="badge bg-light text-dark ms-2">1</span>
                            </div>
                        <div id="invoiceItems">
                                <div class="row invoice-item">
                                <div class="col-md-4">
                                    <select class="form-select" name="product_id[]" required>
                                        <option value="">Select Product</option>
                                        <?php foreach ($products as $product): ?>
                                            <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price']; ?>">
                                                <?php echo htmlspecialchars($product['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="quantity[]" placeholder="Qty" min="1" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="unit_price[]" placeholder="Unit Price" step="0.01" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="total_price[]" placeholder="Total" readonly>
                                </div>
                                <div class="col-md-2">
                                    <div class="d-flex gap-1">
                                        <button type="button" class="btn btn-outline-primary btn-sm add-item">
                                            <i class="fas fa-plus"></i>
                                        </button>
                                        <button type="button" class="btn btn-outline-danger btn-sm remove-item">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Invoice Totals -->
                        <div class="row mt-3">
                            <div class="col-md-8"></div>
                            <div class="col-md-4">
                                <div class="card">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Subtotal:</span>
                                            <span id="subtotalAmount">Ksh 0.00</span>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2 align-items-center">
                                            <div>
                                                <input type="checkbox" id="applyTaxCheckbox" style="margin-right: 5px;">
                                                <label for="applyTaxCheckbox" id="taxLabel">Tax (0%):</label>
                                            </div>
                                            <span id="taxAmount">Ksh 0.00</span>
                                        </div>
                                        <div class="d-flex justify-content-between border-top pt-2">
                                            <strong>Total:</strong>
                                            <strong id="totalAmount">Ksh 0.00</strong>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Create Invoice</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View Invoice Modal -->
    <div class="modal fade" id="viewInvoiceModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-eye me-2"></i>
                        View Invoice
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="invoiceDetails">
                        <div class="text-center">
                            <i class="fas fa-spinner fa-spin fa-2x"></i>
                            <p>Loading invoice details...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="printInvoice">
                        <i class="fas fa-print"></i> Print Invoice
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Payment Modal -->
    <div class="modal fade" id="paymentModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-credit-card me-2"></i>
                        Process Payment
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="paymentForm">
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i>
                            <strong>Invoice:</strong> <span id="paymentInvoiceNumber"></span><br>
                            <strong>Total Amount:</strong> <span id="paymentAmount"></span><br>
                            <strong>Amount Paid:</strong> <span id="paymentAmountPaid">Ksh 0.00</span><br>
                            <strong>Remaining Balance:</strong> <span id="paymentRemainingBalance"></span>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="payment_method" class="form-label">Payment Method</label>
                                    <select class="form-select" id="payment_method" name="payment_method" required>
                                        <option value="">Select Payment Method</option>
                                        <option value="cash">Cash</option>
                                        <?php 
                                        // Check if M-Pesa is enabled
                                        $stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'enable_mpesa' AND setting_group = 'mpesa_settings'");
                                        $stmt->execute();
                                        $mpesaEnabled = $stmt->fetchColumn();
                                        
                                        if ($mpesaEnabled === '1') {
                                            echo '<option value="mpesa">M-Pesa</option>';
                                        }
                                        ?>
                                        <option value="bank_transfer">Bank Transfer</option>
                                        <option value="cheque">Cheque</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="payment_date" class="form-label">Payment Date</label>
                                    <input type="date" class="form-control" id="payment_date" name="payment_date" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="payment_amount" class="form-label">Amount Paid</label>
                                    <input type="number" class="form-control" id="payment_amount" name="payment_amount" step="0.01" required>
                                    <div class="form-text">Enter the amount being paid (partial payments allowed)</div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="reference_number" class="form-label">Reference Number</label>
                                    <input type="text" class="form-control" id="reference_number" name="reference_number" placeholder="Transaction/Cheque number">
                                </div>
                            </div>
                        </div>
                        
                        <!-- M-Pesa Phone Number Field (shown only when M-Pesa is selected) -->
                        <div class="row" id="mpesaPhoneRow" style="display: none;">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="mpesa_phone" class="form-label">M-Pesa Phone Number</label>
                                    <input type="tel" class="form-control" id="mpesa_phone" name="mpesa_phone" placeholder="254XXXXXXXXX or 07XXXXXXXX" pattern="^(254|\+254|0)?[17]\d{8}$">
                                    <div class="form-text">Enter the phone number that will receive the M-Pesa STK push</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="payment_notes" class="form-label">Payment Notes</label>
                            <textarea class="form-control" id="payment_notes" name="payment_notes" rows="2" placeholder="Additional payment details..."></textarea>
                        </div>
                        
                        <input type="hidden" id="payment_invoice_id" name="invoice_id">
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-check"></i> Process Payment
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Invoice Modal -->
    <div class="modal fade" id="editInvoiceModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-edit me-2"></i>
                        Edit Invoice
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="editInvoiceForm">
                    <div class="modal-body">
                        <input type="hidden" id="edit_invoice_id" name="invoice_id">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_customer_id" class="form-label">Customer</label>
                                    <select class="form-select" id="edit_customer_id" name="customer_id" required>
                                        <option value="">Select Customer</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_customer_email" class="form-label">Customer Email</label>
                                    <input type="email" class="form-control" id="edit_customer_email" name="customer_email" readonly>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_issue_date" class="form-label">Issue Date</label>
                                    <input type="date" class="form-control" id="edit_issue_date" name="issue_date" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_due_date" class="form-label">Due Date</label>
                                    <input type="date" class="form-control" id="edit_due_date" name="due_date" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">
                                <i class="fas fa-list me-2"></i>Invoice Items
                                <span class="badge bg-primary" id="editItemCount">0</span>
                            </label>
                            <div class="invoice-items-container" style="max-height: 300px; overflow-y: auto; border: 1px solid #dee2e6; border-radius: 0.375rem; padding: 15px;">
                                <div class="invoice-items-header mb-3">
                                    <div class="row">
                                        <div class="col-md-4"><strong>Product</strong></div>
                                        <div class="col-md-2"><strong>Quantity</strong></div>
                                        <div class="col-md-2"><strong>Unit Price</strong></div>
                                        <div class="col-md-2"><strong>Total</strong></div>
                                        <div class="col-md-2"><strong>Actions</strong></div>
                                    </div>
                                </div>
                                <div id="editInvoiceItems">
                                    <!-- Invoice items will be populated here -->
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_notes" class="form-label">Notes</label>
                                    <textarea class="form-control" id="edit_notes" name="notes" rows="3" placeholder="Additional notes..."></textarea>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <div class="card">
                                        <div class="card-body">
                                            <h6 class="card-title">Invoice Summary</h6>
                                            <div class="d-flex justify-content-between">
                                                <span>Subtotal:</span>
                                                <span id="edit_subtotal">Ksh 0.00</span>
                                            </div>
                                            <div class="d-flex justify-content-between mb-2 align-items-center">
                                                <div>
                                                    <input type="checkbox" id="edit_applyTaxCheckbox" style="margin-right: 5px;">
                                                    <label for="edit_applyTaxCheckbox" id="edit_taxLabel">Tax (0%):</label>
                                                </div>
                                                <span id="edit_tax_amount">Ksh 0.00</span>
                                            </div>
                                            <hr>
                                            <div class="d-flex justify-content-between">
                                                <strong>Total:</strong>
                                                <strong id="edit_total_amount">Ksh 0.00</strong>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Update Invoice
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Custom JavaScript -->
    <script>
        // Output all products as a JS array for dropdown population
        const allProducts = <?php echo json_encode(array_map(function($p) {
            return [
                'id' => $p['id'],
                'name' => $p['name'],
                'price' => $p['price']
            ];
        }, $products)); ?>;

        document.addEventListener('DOMContentLoaded', function() {
            // Set default dates
            const today = new Date();
            const issueDate = document.getElementById('issue_date');
            const dueDate = document.getElementById('due_date');
            
            issueDate.value = today.toISOString().split('T')[0];
            
            const dueDateValue = new Date(today);
            dueDateValue.setDate(dueDateValue.getDate() + 30); // 30 days from today
            dueDate.value = dueDateValue.toISOString().split('T')[0];
            
            // Quotation selection auto-fill and items
            document.getElementById('quotation_id').addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const customerId = selectedOption.dataset.customerId;
                const customerName = selectedOption.dataset.customerName;
                const customerEmail = selectedOption.dataset.customerEmail;
                const customerPhone = selectedOption.dataset.customerPhone;
                if (customerId) {
                    // Set customer dropdown
                    const customerSelect = document.getElementById('customer_id');
                    customerSelect.value = customerId;
                    // Set email field
                    document.getElementById('customer_email').value = customerEmail || '';
                }
                // Fetch and populate quotation items
                if (this.value) {
                    fetch('../api/get_quotation_details.php?id=' + this.value)
                        .then(res => res.json())
                        .then(data => {
                            if (data.success && data.items) {
                                const items = data.items;
                                const invoiceItems = document.getElementById('invoiceItems');
                                invoiceItems.innerHTML = '';
                                items.forEach(item => {
                                    const row = document.createElement('div');
                                    row.className = 'row invoice-item mb-2';
                                    row.innerHTML = `
                                        <div class="col-md-4">
                                            <select class="form-select" name="product_id[]" required>
                                                <option value="">Select Product</option>
                                                ${allProducts.map(opt =>
                                                    `<option value="${opt.id}" data-price="${opt.price}" ${opt.id == item.product_id ? 'selected' : ''}>${opt.name}</option>`
                                                ).join('')}
                                            </select>
                                        </div>
                                        <div class="col-md-2">
                                            <input type="number" class="form-control" name="quantity[]" placeholder="Qty" min="1" required value="${item.quantity}">
                                        </div>
                                        <div class="col-md-2">
                                            <input type="number" class="form-control" name="unit_price[]" placeholder="Unit Price" step="0.01" required value="${item.unit_price}">
                                        </div>
                                        <div class="col-md-2">
                                            <input type="number" class="form-control" name="total_price[]" placeholder="Total" readonly value="${item.total_price}">
                                        </div>
                                        <div class="col-md-2">
                                            <div class="d-flex gap-1">
                                                <button type="button" class="btn btn-outline-primary btn-sm add-item"><i class="fas fa-plus"></i></button>
                                                <button type="button" class="btn btn-outline-danger btn-sm remove-item"><i class="fas fa-trash"></i></button>
                                            </div>
                                        </div>
                                    `;
                                    invoiceItems.appendChild(row);
                                });
                                updateItemEventListeners();
                                calculateTotals();
                            }
                        });
                }
            });
            
            // Customer selection change
            document.getElementById('customer_id').addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const emailInput = document.getElementById('customer_email');
                
                if (selectedOption.dataset.email) {
                    emailInput.value = selectedOption.dataset.email;
                }
            });
            
            // Add/Remove item functionality
            document.addEventListener('click', function(e) {
                if (e.target.closest('.add-item')) {
                    const invoiceItems = document.getElementById('invoiceItems');
                    const newItem = invoiceItems.querySelector('.invoice-item').cloneNode(true);
                    
                    // Clear values
                    newItem.querySelectorAll('input, select').forEach(input => {
                        input.value = '';
                    });
                    
                    invoiceItems.appendChild(newItem);
                    updateItemEventListeners();
                    
                    // Update item counter
                    updateItemCounter();
                    
                    // Scroll to the new item if there are many items
                    const itemCount = document.querySelectorAll('.invoice-item').length;
                    if (itemCount > 8) {
                        newItem.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    }
                }
                
                if (e.target.closest('.remove-item')) {
                    if (document.querySelectorAll('.invoice-item').length > 1) {
                        e.target.closest('.invoice-item').remove();
                        calculateTotals();
                        updateItemCounter();
                    }
                }
            });
            
            function updateItemEventListeners() {
                // Product selection change
                document.querySelectorAll('select[name="product_id[]"]').forEach(select => {
                    select.addEventListener('change', function() {
                        const row = this.closest('.invoice-item');
                        const priceInput = row.querySelector('input[name="unit_price[]"]');
                        const selectedOption = this.options[this.selectedIndex];
                        
                        if (selectedOption.dataset.price) {
                            priceInput.value = selectedOption.dataset.price;
                            calculateRowTotal(row);
                        }
                    });
                });
                
                // Quantity and price change
                document.querySelectorAll('input[name="quantity[]"], input[name="unit_price[]"]').forEach(input => {
                    input.addEventListener('input', function() {
                        calculateRowTotal(this.closest('.invoice-item'));
                    });
                });
            }
            
            function calculateRowTotal(row) {
                const quantity = parseFloat(row.querySelector('input[name="quantity[]"]').value) || 0;
                const unitPrice = parseFloat(row.querySelector('input[name="unit_price[]"]').value) || 0;
                const totalInput = row.querySelector('input[name="total_price[]"]');
                
                totalInput.value = (quantity * unitPrice).toFixed(2);
                calculateTotals();
            }
            
            function calculateTotals() {
                let subtotal = 0;
                document.querySelectorAll('input[name="total_price[]"]').forEach(input => {
                    subtotal += parseFloat(input.value) || 0;
                });
                
                const tax = subtotal * 0.00; // 0% tax
                const total = subtotal + tax;
                
                document.getElementById('subtotalAmount').textContent = 'Ksh ' + subtotal.toFixed(2);
                document.getElementById('taxAmount').textContent = 'Ksh ' + tax.toFixed(2);
                document.getElementById('totalAmount').textContent = 'Ksh ' + total.toFixed(2);
            }



            function recalculateTotals() {
                let subtotal = 0;
                document.querySelectorAll('input[name="total_price[]"]').forEach(function(input) {
                    subtotal += parseFloat(input.value) || 0;
                });
                document.getElementById('subtotalAmount').textContent = 'Ksh ' + subtotal.toFixed(2);
                
                // Tax logic
                const taxCheckbox = document.getElementById('applyTaxCheckbox');
                let taxRate = taxCheckbox.checked ? 0.16 : 0;
                let taxLabel = taxCheckbox.checked ? 'Tax (16%):' : 'Tax (0%):';
                document.getElementById('taxLabel').textContent = taxLabel;
                let taxAmount = subtotal * taxRate;
                document.getElementById('taxAmount').textContent = 'Ksh ' + taxAmount.toFixed(2);
                
                let total = subtotal + taxAmount;
                document.getElementById('totalAmount').textContent = 'Ksh ' + total.toFixed(2);
            }

            // Attach event listener to checkbox
            if (document.getElementById('applyTaxCheckbox')) {
                document.getElementById('applyTaxCheckbox').addEventListener('change', recalculateTotals);
            }
            
            // Initialize event listeners
            updateItemEventListeners();
            
            // Form submission
            document.getElementById('addInvoiceForm').addEventListener('submit', function(e) {
                e.preventDefault();
                
                // Validate form
                const form = e.target;
                const formData = new FormData(form);
                
                // Validate that at least one product is selected
                const productSelects = form.querySelectorAll('select[name="product_id[]"]');
                let hasValidItems = false;
                
                productSelects.forEach(select => {
                    if (select.value) {
                        hasValidItems = true;
                    }
                });
                
                if (!hasValidItems) {
                    alert('Please select at least one product');
                    return;
                }
                
                // Show loading state
                const submitBtn = form.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creating...';
                
                // Submit to API
                fetch('../api/create_invoice.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert(data.message);
                        
                        // Reset form
                        form.reset();
                        
                        // Reset items to just one row
                        const invoiceItems = document.getElementById('invoiceItems');
                        invoiceItems.innerHTML = `
                            <div class="row invoice-item mb-2">
                                <div class="col-md-4">
                                    <select class="form-select" name="product_id[]" required>
                                        <option value="">Select Product</option>
                                        ${allProducts.map(opt =>
                                            `<option value="${opt.id}" data-price="${opt.price}">${opt.name}</option>`
                                        ).join('')}
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="quantity[]" placeholder="Qty" min="1" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="unit_price[]" placeholder="Unit Price" step="0.01" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="total_price[]" placeholder="Total" readonly>
                                </div>
                                <div class="col-md-2">
                                    <div class="d-flex gap-1">
                                        <button type="button" class="btn btn-outline-primary btn-sm add-item"><i class="fas fa-plus"></i></button>
                                        <button type="button" class="btn btn-outline-danger btn-sm remove-item"><i class="fas fa-trash"></i></button>
                                    </div>
                                </div>
                            </div>
                        `;
                        
                        // Reset totals
                        document.getElementById('subtotalAmount').textContent = 'Ksh 0.00';
                        document.getElementById('taxAmount').textContent = 'Ksh 0.00';
                        document.getElementById('totalAmount').textContent = 'Ksh 0.00';
                        
                        // Reset dates
                        const today = new Date();
                        const issueDate = document.getElementById('issue_date');
                        const dueDate = document.getElementById('due_date');
                        
                        issueDate.value = today.toISOString().split('T')[0];
                        
                        const dueDateValue = new Date(today);
                        dueDateValue.setDate(dueDateValue.getDate() + 30);
                        dueDate.value = dueDateValue.toISOString().split('T')[0];
                        
                        // Re-initialize event listeners
                        updateItemEventListeners();
                        
                        // Close modal
                bootstrap.Modal.getInstance(document.getElementById('addInvoiceModal')).hide();
                        
                        // Refresh the page to show the new invoice
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                        
                    } else {
                        alert(data.message || 'Error creating invoice');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Network error. Please try again.');
                })
                .finally(() => {
                    // Reset button state
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                });
            });
            
            // Payment method change handler
            document.getElementById('payment_method').addEventListener('change', function() {
                const mpesaPhoneRow = document.getElementById('mpesaPhoneRow');
                const mpesaPhoneInput = document.getElementById('mpesa_phone');
                
                if (this.value === 'mpesa') {
                    mpesaPhoneRow.style.display = 'block';
                    mpesaPhoneInput.required = true;
                } else {
                    mpesaPhoneRow.style.display = 'none';
                    mpesaPhoneInput.required = false;
                    mpesaPhoneInput.value = '';
                }
            });
            
            // Payment form submission
            document.getElementById('paymentForm').addEventListener('submit', function(e) {
                e.preventDefault();
                
                // Validate form
                const form = e.target;
                const formData = new FormData(form);
                
                // Show loading state
                const submitBtn = form.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
                
                // Check if M-Pesa is selected
                const paymentMethod = formData.get('payment_method');
                const mpesaPhone = formData.get('mpesa_phone');
                
                if (paymentMethod === 'mpesa') {
                    // Validate phone number
                    if (!mpesaPhone || mpesaPhone.trim() === '') {
                        alert('Please enter a valid M-Pesa phone number');
                        return;
                    }
                    
                    // Process M-Pesa STK Push
                    const stkPushData = {
                        amount: parseFloat(formData.get('payment_amount')),
                        phone: mpesaPhone.trim(),
                        customer_id: formData.get('customer_id') || null,
                        customer_name: formData.get('customer_name') || 'Invoice Payment',
                        invoice_id: formData.get('invoice_id')
                    };
                    
                    fetch('../api/mpesa_invoice_stkpush.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify(stkPushData)
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            alert('M-Pesa STK Push sent successfully! Please check your phone and complete the payment.');
                            
                            // Close modal
                            bootstrap.Modal.getInstance(document.getElementById('paymentModal')).hide();
                            
                            // Refresh the page after a delay to check payment status
                            setTimeout(() => {
                                location.reload();
                            }, 3000);
                            
                        } else {
                            alert('M-Pesa Error: ' + (data.message || 'Failed to send STK push'));
                        }
                    })
                    .catch(error => {
                        console.error('M-Pesa Error:', error);
                        alert('Network error. Please try again.');
                    })
                    .finally(() => {
                        // Reset button state
                        submitBtn.disabled = false;
                        submitBtn.textContent = originalText;
                    });
                    
                } else {
                    // Process regular payment
                    fetch('../api/process_payment.php', {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            alert(data.message);
                            // Update modal fields immediately
                            const paidSpan = document.getElementById('paymentAmountPaid');
                            const amountSpan = document.getElementById('paymentAmount');
                            const balanceSpan = document.getElementById('paymentRemainingBalance');
                            if (paidSpan && amountSpan && balanceSpan) {
                                paidSpan.textContent = 'Ksh ' + (data.paid ? parseFloat(data.paid).toFixed(2) : '0.00');
                                amountSpan.textContent = 'Ksh ' + (data.amount ? parseFloat(data.amount).toFixed(2) : '0.00');
                                const balance = (data.amount ? parseFloat(data.amount) : 0) - (data.paid ? parseFloat(data.paid) : 0);
                                balanceSpan.textContent = 'Ksh ' + balance.toFixed(2);
                            }
                            // Close modal
                            bootstrap.Modal.getInstance(document.getElementById('paymentModal')).hide();
                            // Refresh the page to show updated status
                            setTimeout(() => {
                                location.reload();
                            }, 1500);
                        } else {
                            alert(data.message || 'Error processing payment');
                        }
                    })
                                    .catch(error => {
                        console.error('Error:', error);
                        alert('Network error. Please try again.');
                    })
                    .finally(() => {
                        // Reset button state
                        submitBtn.disabled = false;
                        submitBtn.textContent = originalText;
                    });
                }
            });
        });

        // Global functions
        function updateItemCounter() {
            const itemCount = document.querySelectorAll('.invoice-item').length;
            const counterElement = document.getElementById('itemCount');
            if (counterElement) {
                counterElement.textContent = itemCount;
                
                // Change badge color based on item count
                counterElement.className = 'badge ms-2';
                if (itemCount >= 8) {
                    counterElement.classList.add('bg-warning', 'text-dark');
                } else if (itemCount >= 5) {
                    counterElement.classList.add('bg-info', 'text-white');
                } else {
                    counterElement.classList.add('bg-light', 'text-dark');
                }
            }
        }

        document.addEventListener('DOMContentLoaded', function() {
            // Initialize item counter
            updateItemCounter();
            
            const searchInput = document.getElementById('invoiceSearchInput');
            if (searchInput) {
                let searchTimeout;
                searchInput.addEventListener('input', function() {
                    clearTimeout(searchTimeout);
                    const query = this.value.trim();
                    
                    // Show loading indicator
                    const tbody = document.querySelector('.table tbody');
                    if (query.length > 0) {
                        tbody.innerHTML = '<tr><td colspan="9" class="text-center"><i class="fas fa-spinner fa-spin"></i> Searching...</td></tr>';
                    }
                    
                    // Debounce search requests
                    searchTimeout = setTimeout(() => {
                        fetch(`../api/search_invoices.php?q=${encodeURIComponent(query)}`)
                            .then(res => res.json())
                            .then(data => {
                                if (data.success) {
                                    updateInvoiceTable(data.invoices);
                                } else {
                                    tbody.innerHTML = '<tr><td colspan="7" class="text-center text-danger">Error: ' + (data.error || 'Search failed') + '</td></tr>';
                                }
                            })
                            .catch(error => {
                                tbody.innerHTML = '<tr><td colspan="7" class="text-center text-danger">Error: ' + error.message + '</td></tr>';
                            });
                    }, 300);
                });
            }
            
            function updateInvoiceTable(invoices) {
                const tbody = document.querySelector('.table tbody');
                
                if (invoices.length === 0) {
                    tbody.innerHTML = '<tr><td colspan="9" class="text-center"><i class="fas fa-search fa-2x text-muted mb-2"></i><p class="text-muted">No invoices found matching your search.</p></td></tr>';
                    return;
                }
                
                tbody.innerHTML = invoices.map(invoice => {
                    const totalPaid = parseFloat(invoice.total_paid || 0);
                    const balance = parseFloat(invoice.total_amount) - totalPaid;
                    
                    return `
                        <tr>
                            <td>${escapeHtml(invoice.invoice_number)}</td>
                            <td>${escapeHtml(invoice.customer_name || 'Walk-in Customer')}</td>
                            <td>Ksh ${parseFloat(invoice.total_amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                            <td>Ksh ${totalPaid.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                            <td>
                                ${balance > 0 ? 
                                    `<span class="text-danger">Ksh ${balance.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>` :
                                    `<span class="text-success">Ksh 0.00</span>`
                                }
                            </td>
                            <td>
                                <span class="badge ${getStatusBadgeClass(invoice.status)}">
                                    ${invoice.status.charAt(0).toUpperCase() + invoice.status.slice(1)}
                                </span>
                            </td>
                            <td>${formatDate(invoice.issue_date)}</td>
                            <td>${formatDate(invoice.due_date)}</td>
                            <td>
                                <div class="btn-group">
                                    <button class="btn btn-sm btn-outline-primary view-invoice" title="View Invoice" data-id="${invoice.id}" data-invoice-number="${escapeHtml(invoice.invoice_number)}">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    ${invoice.status !== 'paid' ? `
                                    <button class="btn btn-sm btn-outline-info edit-invoice" title="Edit" data-id="${invoice.id}" data-invoice-number="${escapeHtml(invoice.invoice_number)}">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    ` : ''}
                                    ${balance > 0 ? `
                                    <button class="btn btn-sm btn-outline-success pay-invoice" title="Pay Invoice" data-id="${invoice.id}" data-invoice-number="${escapeHtml(invoice.invoice_number)}" data-amount="${invoice.total_amount}">
                                        <i class="fas fa-credit-card"></i>
                                    </button>
                                    ` : ''}
                                    <button class="btn btn-sm btn-outline-secondary print-invoice" title="Print Invoice" data-id="${invoice.id}" data-invoice-number="${escapeHtml(invoice.invoice_number)}">
                                        <i class="fas fa-print"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-warning send-reminder" title="Send WhatsApp Reminder" data-id="${invoice.id}" data-invoice-number="${escapeHtml(invoice.invoice_number)}">
                                        <i class="fas fa-whatsapp"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-danger delete-invoice" title="Delete" data-id="${invoice.id}" data-invoice-number="${escapeHtml(invoice.invoice_number)}">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    `;
                }).join('');
            }
            
            function getStatusBadgeClass(status) {
                switch(status) {
                    case 'paid': return 'bg-success';
                    case 'pending': return 'bg-warning';
                    case 'overdue': return 'bg-danger';
                    case 'cancelled': return 'bg-secondary';
                    default: return 'bg-info';
                }
            }
            
            function formatDate(dateString) {
                return new Date(dateString).toLocaleDateString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    year: 'numeric'
                });
            }
            
            function escapeHtml(text) {
                const div = document.createElement('div');
                div.textContent = text;
                return div.innerHTML;
            }
            
            // Action button handlers
            document.addEventListener('click', function(e) {
                // View invoice handler
                if (e.target.closest('.view-invoice')) {
                    const btn = e.target.closest('.view-invoice');
                    const invoiceId = btn.dataset.id;
                    const invoiceNumber = btn.dataset.invoiceNumber;
                    
                    // Update modal title
                    document.querySelector('#viewInvoiceModal .modal-title').textContent = 'View Invoice - ' + invoiceNumber;
                    
                    // Set invoice ID on print button
                    document.getElementById('printInvoice').setAttribute('data-invoice-id', invoiceId);
                    
                    // Load invoice details
                    fetch(`../api/get_invoice_details.php?id=${invoiceId}`)
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                displayInvoiceDetails(data.invoice, data.items);
                            } else {
                                document.getElementById('invoiceDetails').innerHTML = 
                                    '<div class="alert alert-danger">Error loading invoice details: ' + data.message + '</div>';
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            document.getElementById('invoiceDetails').innerHTML = 
                                '<div class="alert alert-danger">Network error. Please try again.</div>';
                        });
                    
                    // Show modal
                    const viewModal = new bootstrap.Modal(document.getElementById('viewInvoiceModal'));
                    viewModal.show();
                }
                
                // Pay invoice handler
                if (e.target.closest('.pay-invoice')) {
                    const btn = e.target.closest('.pay-invoice');
                    const invoiceId = btn.dataset.id;
                    const invoiceNumber = btn.dataset.invoiceNumber;
                    const amount = btn.dataset.amount;

                    // Fetch payment history for this invoice
                    fetch(`../api/get_invoice_payments.php?id=${invoiceId}`)
                        .then(response => response.json())
                        .then(data => {
                            let totalPaid = 0;
                            if (data.success && data.payments) {
                                totalPaid = data.payments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0);
                            }
                            const remainingBalance = parseFloat(amount) - totalPaid;

                            // Populate payment modal
                            document.getElementById('paymentInvoiceNumber').textContent = invoiceNumber;
                            document.getElementById('paymentAmount').textContent = 'Ksh ' + parseFloat(amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                            document.getElementById('paymentAmountPaid').textContent = 'Ksh ' + totalPaid.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                            document.getElementById('paymentRemainingBalance').textContent = 'Ksh ' + remainingBalance.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
                            document.getElementById('payment_invoice_id').value = invoiceId;
                            document.getElementById('payment_amount').value = remainingBalance > 0 ? remainingBalance.toFixed(2) : amount;
                            document.getElementById('payment_date').value = new Date().toISOString().split('T')[0];

                            // Show payment modal ONLY after updating fields
                            const paymentModal = new bootstrap.Modal(document.getElementById('paymentModal'));
                            paymentModal.show();
                        })
                        .catch(error => {
                            alert('Error fetching payment history: ' + error);
                        });
                }
                
                // Edit invoice handler
                if (e.target.closest('.edit-invoice')) {
                    const btn = e.target.closest('.edit-invoice');
                    const invoiceId = btn.dataset.id;
                    const invoiceNumber = btn.dataset.invoiceNumber;
                    
                    // Show loading state
                    btn.disabled = true;
                    const originalContent = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                    
                    // Fetch invoice details for editing
                    fetch(`../api/get_invoice_for_edit.php?id=${invoiceId}`)
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                populateEditModal(data);
                                const editModal = new bootstrap.Modal(document.getElementById('editInvoiceModal'));
                                editModal.show();
                            } else {
                                alert('Error loading invoice: ' + data.message);
                            }
                        })
                        .catch(error => {
                            console.error('Error fetching invoice details:', error);
                            alert('Error loading invoice details. Please try again.');
                        })
                        .finally(() => {
                            // Reset button state
                            btn.disabled = false;
                            btn.innerHTML = originalContent;
                        });
                }
                
                // Send reminder handler
                if (e.target.closest('.send-reminder')) {
                    const btn = e.target.closest('.send-reminder');
                    const invoiceId = btn.dataset.id;
                    const invoiceNumber = btn.dataset.invoiceNumber;
                    
                    // Show loading state
                    btn.disabled = true;
                    const originalContent = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                    
                    // First generate PDF
                    fetch(`../api/generate_invoice_pdf_fixed.php?id=${invoiceId}`)
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                // Get customer phone number
                                fetch(`../api/get_invoice_details.php?id=${invoiceId}`)
                                    .then(response => response.json())
                                    .then(invoiceData => {
                                        if (invoiceData.success && invoiceData.invoice.customer_phone) {
                                            const phone = invoiceData.invoice.customer_phone;
                                            const formattedPhone = phone.replace(/^0/, '254').replace(/^\+/, '');
                                            
                                            // Create WhatsApp message
                                            const message = `Hello! Here is your invoice ${invoiceNumber} from ${invoiceData.invoice.store_name || 'Our Store'}. Please find the PDF attached. Total amount: Ksh ${parseFloat(invoiceData.invoice.total_amount).toLocaleString()}. Thank you for your business!`;
                                            
                                            // Create WhatsApp link
                                            const whatsappUrl = `https://wa.me/${formattedPhone}?text=${encodeURIComponent(message)}`;
                                            
                                            // Open WhatsApp in new tab
                                            window.open(whatsappUrl, '_blank');
                                            
                                            // Show appropriate message based on file type
                                            if (data.is_pdf) {
                                                alert(`WhatsApp opened for invoice ${invoiceNumber}. Please attach the generated PDF file manually.`);
                                            } else {
                                                alert(`WhatsApp opened for invoice ${invoiceNumber}. The invoice was generated as HTML. Please print it as PDF first, then attach to WhatsApp.`);
                                            }
                                        } else {
                                            alert('Customer phone number not found. Cannot send WhatsApp reminder.');
                                        }
                                    })
                                    .catch(error => {
                                        console.error('Error fetching invoice details:', error);
                                        alert('Error fetching invoice details. Please try again.');
                                    });
                            } else {
                                alert('Error generating PDF: ' + data.message);
                            }
                        })
                        .catch(error => {
                            console.error('Error generating PDF:', error);
                            alert('Error generating PDF. Please try again.');
                        })
                        .finally(() => {
                            // Reset button state
                            btn.disabled = false;
                            btn.innerHTML = originalContent;
                        });
                }
                
                // Print invoice handler (from table)
                if (e.target.closest('.print-invoice')) {
                    const btn = e.target.closest('.print-invoice');
                    const invoiceId = btn.dataset.id;
                    const invoiceNumber = btn.dataset.invoiceNumber;
                    
                    // Show loading state
                    btn.disabled = true;
                    const originalContent = btn.innerHTML;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                    
                    // Open print invoice page in new tab
                    window.open(`../receipts/print-invoice.php?id=${invoiceId}`, '_blank');
                    
                    // Reset button state
                    btn.disabled = false;
                    btn.innerHTML = originalContent;
                }
                
                // Delete invoice handler
                if (e.target.closest('.delete-invoice')) {
                    const btn = e.target.closest('.delete-invoice');
                    const invoiceId = btn.dataset.id;
                    const invoiceNumber = btn.dataset.invoiceNumber;
                    
                    if (confirm('Are you sure you want to delete invoice ' + invoiceNumber + '? This action cannot be undone.')) {
                        alert('Delete functionality for invoice ' + invoiceNumber + ' will be implemented.');
                    }
                }
            });
            
            // Print invoice functionality
            document.getElementById('printInvoice').addEventListener('click', function() {
                const invoiceId = this.getAttribute('data-invoice-id');
                if (invoiceId) {
                    // Show loading state
                    const btn = this;
                    const originalText = btn.innerHTML;
                    btn.disabled = true;
                    btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Generating...';
                    
                    // Open print invoice page in new tab
                    window.open(`../receipts/print-invoice.php?id=${invoiceId}`, '_blank');
                    
                    // Reset button state
                    btn.disabled = false;
                    btn.innerHTML = originalText;
                } else {
                    alert('Invoice ID not found. Please try viewing the invoice again.');
                }
            });
            
            // Function to display invoice details in modal
            function displayInvoiceDetails(invoice, items) {
                const customerInfo = invoice.customer_name ? `
                    <div class="detail-row">
                        <span class="detail-label">Customer:</span>
                        <span class="detail-value">${invoice.customer_name}</span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">Phone:</span>
                        <span class="detail-value">${invoice.customer_phone || 'N/A'}</span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">Email:</span>
                        <span class="detail-value">${invoice.customer_email || 'N/A'}</span>
                    </div>
                    <div class="detail-row">
                        <span class="detail-label">Address:</span>
                        <span class="detail-value">${invoice.customer_address || 'N/A'}</span>
                    </div>
                ` : '<div class="detail-row"><span class="detail-label">Customer:</span><span class="detail-value">Walk-in Customer</span></div>';
                
                let itemsHtml = '';
                items.forEach(item => {
                    itemsHtml += `
                        <tr>
                            <td>${item.product_name}</td>
                            <td>${item.quantity}</td>
                            <td>Ksh ${parseFloat(item.unit_price).toFixed(2)}</td>
                            <td>Ksh ${parseFloat(item.total_price).toFixed(2)}</td>
                        </tr>
                    `;
                });
                
                const html = `
                    <style>
                        .invoice-info {
                            display: grid;
                            grid-template-columns: 1fr 1fr;
                            gap: 30px;
                            margin-bottom: 30px;
                        }
                        
                        .invoice-details, .customer-details {
                            background-color: #f8f9fa;
                            padding: 20px;
                            border-radius: 8px;
                            border-left: 4px solid #6f42c1;
                        }
                        
                        .invoice-details h3, .customer-details h3 {
                            color: #6f42c1;
                            border-bottom: 2px solid #6f42c1;
                            padding-bottom: 10px;
                            margin-bottom: 15px;
                        }
                        
                        .detail-row {
                            display: flex;
                            justify-content: space-between;
                            margin-bottom: 8px;
                        }
                        
                        .detail-label {
                            font-weight: bold;
                            color: #666;
                        }
                        
                        .detail-value {
                            color: #333;
                        }
                        
                        .items-table {
                            width: 100%;
                            border-collapse: collapse;
                            margin: 20px 0;
                        }
                        
                        .items-table th {
                            background-color: #6f42c1;
                            color: white;
                            padding: 12px;
                            text-align: left;
                        }
                        
                        .items-table td {
                            padding: 12px;
                            border-bottom: 1px solid #ddd;
                        }
                        
                        .items-table tr:nth-child(even) {
                            background-color: #f8f9fa;
                        }
                        
                        .totals-section {
                            margin-top: 30px;
                            text-align: right;
                        }
                        
                        .total-row {
                            display: flex;
                            justify-content: space-between;
                            margin-bottom: 10px;
                            font-size: 1.1em;
                        }
                        
                        .total-row.grand-total {
                            font-size: 1.3em;
                            font-weight: bold;
                            color: #6f42c1;
                            border-top: 2px solid #6f42c1;
                            padding-top: 10px;
                        }
                    </style>
                    
                    <div class="invoice-info">
                        <div class="invoice-details">
                            <h3>Invoice Details</h3>
                            <div class="detail-row">
                                <span class="detail-label">Invoice Number:</span>
                                <span class="detail-value">${invoice.invoice_number}</span>
                            </div>
                            <div class="detail-row">
                                <span class="detail-label">Status:</span>
                                <span class="detail-value">
                                    <span class="badge ${getStatusBadgeClass(invoice.status)}">
                                        ${invoice.status.charAt(0).toUpperCase() + invoice.status.slice(1)}
                                    </span>
                                </span>
                            </div>
                            <div class="detail-row">
                                <span class="detail-label">Issue Date:</span>
                                <span class="detail-value">${new Date(invoice.issue_date).toLocaleDateString()}</span>
                            </div>
                            <div class="detail-row">
                                <span class="detail-label">Due Date:</span>
                                <span class="detail-value">${new Date(invoice.due_date).toLocaleDateString()}</span>
                            </div>
                            <div class="detail-row">
                                <span class="detail-label">Created By:</span>
                                <span class="detail-value">${invoice.created_by_name}</span>
                            </div>
                            ${invoice.quotation_reference ? `
                            <div class="detail-row">
                                <span class="detail-label">Quotation Ref:</span>
                                <span class="detail-value">${invoice.quotation_reference}</span>
                            </div>
                            ` : ''}
                        </div>
                        
                        <div class="customer-details">
                            <h3>Customer Information</h3>
                            ${customerInfo}
                        </div>
                    </div>
                    
                    
                    <table class="items-table">
                        <thead>
                            <tr>
                                <th>Item</th>
                                <th>Quantity</th>
                                <th>Unit Price</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${itemsHtml}
                        </tbody>
                    </table>
                    
                    <div class="totals-section">
                        <div class="total-row">
                            <span>Subtotal:</span>
                            <span>Ksh ${parseFloat(invoice.subtotal).toFixed(2)}</span>
                        </div>
                        <div class="total-row">
                            <span>Tax (0%):</span>
                            <span>Ksh ${parseFloat(invoice.tax_amount).toFixed(2)}</span>
                        </div>
                        <div class="total-row grand-total">
                            <span>Total Amount:</span>
                            <span>Ksh ${parseFloat(invoice.total_amount).toFixed(2)}</span>
                        </div>
                    </div>
                    
                    ${invoice.notes ? `
                        <div style="margin-top: 30px; padding: 20px; background-color: #f8f9fa; border-radius: 5px;">
                            <h4 style="color: #6f42c1; margin-bottom: 10px;">Notes</h4>
                            <p style="margin: 0; color: #666;">${invoice.notes}</p>
                        </div>
                    ` : ''}
                `;
                
                document.getElementById('invoiceDetails').innerHTML = html;
            }
            
            // Edit Invoice Functions
            function populateEditModal(data) {
                const invoice = data.invoice;
                const items = data.items;
                const customers = data.customers;
                const products = data.products;
                
                // Set invoice ID
                document.getElementById('edit_invoice_id').value = invoice.id;
                
                // Populate customer dropdown
                const customerSelect = document.getElementById('edit_customer_id');
                customerSelect.innerHTML = '<option value="">Select Customer</option>';
                customers.forEach(customer => {
                    const option = document.createElement('option');
                    option.value = customer.id;
                    option.textContent = customer.name;
                    option.dataset.email = customer.email || '';
                    if (customer.id == invoice.customer_id) {
                        option.selected = true;
                    }
                    customerSelect.appendChild(option);
                });
                
                // Set customer email
                const selectedCustomer = customers.find(c => c.id == invoice.customer_id);
                document.getElementById('edit_customer_email').value = selectedCustomer ? selectedCustomer.email : '';
                
                // Set dates
                document.getElementById('edit_issue_date').value = invoice.issue_date;
                document.getElementById('edit_due_date').value = invoice.due_date;
                
                // Set notes
                document.getElementById('edit_notes').value = invoice.notes || '';
                
                // Set tax checkbox based on original invoice tax amount
                const taxCheckbox = document.getElementById('edit_applyTaxCheckbox');
                const originalTaxAmount = parseFloat(invoice.tax_amount) || 0;
                const originalSubtotal = parseFloat(invoice.subtotal) || 0;
                
                if (originalTaxAmount > 0 && originalSubtotal > 0) {
                    const originalTaxRate = originalTaxAmount / originalSubtotal;
                    taxCheckbox.checked = originalTaxRate > 0;
                } else {
                    taxCheckbox.checked = false;
                }
                
                // Populate invoice items
                populateEditInvoiceItems(items, products);
                
                // Calculate totals
                calculateEditTotals();
            }
            
            function populateEditInvoiceItems(items, products) {
                const container = document.getElementById('editInvoiceItems');
                container.innerHTML = '';
                
                items.forEach((item, index) => {
                    const row = document.createElement('div');
                    row.className = 'row edit-invoice-item mb-2';
                    row.innerHTML = `
                        <div class="col-md-4">
                            <select class="form-select edit-product-select" name="product_id[]" required>
                                <option value="">Select Product</option>
                                ${products.map(opt =>
                                    `<option value="${opt.id}" data-price="${opt.price}" ${opt.id == item.product_id ? 'selected' : ''}>${opt.name}</option>`
                                ).join('')}
                            </select>
                        </div>
                        <div class="col-md-2">
                            <input type="number" class="form-control edit-quantity" name="quantity[]" placeholder="Qty" min="1" required value="${item.quantity}">
                        </div>
                        <div class="col-md-2">
                            <input type="number" class="form-control edit-unit-price" name="unit_price[]" placeholder="Unit Price" step="0.01" required value="${item.unit_price}">
                        </div>
                        <div class="col-md-2">
                            <input type="number" class="form-control edit-total-price" name="total_price[]" placeholder="Total" readonly value="${item.total_price}">
                        </div>
                        <div class="col-md-2">
                            <div class="d-flex gap-1">
                                <button type="button" class="btn btn-outline-primary btn-sm edit-add-item"><i class="fas fa-plus"></i></button>
                                <button type="button" class="btn btn-outline-danger btn-sm edit-remove-item"><i class="fas fa-trash"></i></button>
                            </div>
                        </div>
                    `;
                    container.appendChild(row);
                });
                
                // Add event listeners for edit items
                updateEditItemEventListeners();
                updateEditItemCounter();
            }
            
            function updateEditItemEventListeners() {
                // Product selection change
                document.querySelectorAll('.edit-product-select').forEach(select => {
                    select.addEventListener('change', function() {
                        const row = this.closest('.edit-invoice-item');
                        const priceInput = row.querySelector('.edit-unit-price');
                        const selectedOption = this.options[this.selectedIndex];
                        
                        if (selectedOption.dataset.price) {
                            priceInput.value = selectedOption.dataset.price;
                        }
                        
                        calculateEditItemTotal(row);
                        calculateEditTotals();
                    });
                });
                
                // Quantity and unit price change
                document.querySelectorAll('.edit-quantity, .edit-unit-price').forEach(input => {
                    input.addEventListener('input', function() {
                        const row = this.closest('.edit-invoice-item');
                        calculateEditItemTotal(row);
                        calculateEditTotals();
                    });
                });
                
                // Add/Remove item buttons
                document.querySelectorAll('.edit-add-item').forEach(btn => {
                    btn.addEventListener('click', function() {
                        const container = document.getElementById('editInvoiceItems');
                        const firstItem = container.querySelector('.edit-invoice-item');
                        const newItem = firstItem.cloneNode(true);
                        
                        // Clear values
                        newItem.querySelectorAll('input, select').forEach(input => {
                            input.value = '';
                        });
                        
                        container.appendChild(newItem);
                        updateEditItemEventListeners();
                        updateEditItemCounter();
                    });
                });
                
                document.querySelectorAll('.edit-remove-item').forEach(btn => {
                    btn.addEventListener('click', function() {
                        if (document.querySelectorAll('.edit-invoice-item').length > 1) {
                            this.closest('.edit-invoice-item').remove();
                            calculateEditTotals();
                            updateEditItemCounter();
                        }
                    });
                });
            }
            
            function calculateEditItemTotal(row) {
                const quantity = parseFloat(row.querySelector('.edit-quantity').value) || 0;
                const unitPrice = parseFloat(row.querySelector('.edit-unit-price').value) || 0;
                const totalPrice = quantity * unitPrice;
                row.querySelector('.edit-total-price').value = totalPrice.toFixed(2);
            }
            
            function calculateEditTotals() {
                let subtotal = 0;
                
                document.querySelectorAll('.edit-invoice-item').forEach(row => {
                    const totalPrice = parseFloat(row.querySelector('.edit-total-price').value) || 0;
                    subtotal += totalPrice;
                });
                
                // Tax logic
                const taxCheckbox = document.getElementById('edit_applyTaxCheckbox');
                let taxRate = taxCheckbox.checked ? 0.16 : 0;
                let taxLabel = taxCheckbox.checked ? 'Tax (16%):' : 'Tax (0%):';
                document.getElementById('edit_taxLabel').textContent = taxLabel;
                let taxAmount = subtotal * taxRate;
                const totalAmount = subtotal + taxAmount;
                
                document.getElementById('edit_subtotal').textContent = 'Ksh ' + subtotal.toFixed(2);
                document.getElementById('edit_tax_amount').textContent = 'Ksh ' + taxAmount.toFixed(2);
                document.getElementById('edit_total_amount').textContent = 'Ksh ' + totalAmount.toFixed(2);
            }
            
            function updateEditItemCounter() {
                const count = document.querySelectorAll('.edit-invoice-item').length;
                document.getElementById('editItemCount').textContent = count;
            }
            
            // Edit form submission
            const editForm = document.getElementById('editInvoiceForm');
            if (!editForm) {
                console.error('Edit form not found!');
                return;
            }
            
            editForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                try {
                    console.log('Edit form submitted');
                    
                    const formData = new FormData(this);
                    const invoiceId = formData.get('invoice_id');
                    
                    console.log('Invoice ID:', invoiceId);
                    
                    // Check if all required elements exist
                    const requiredElements = [
                        'edit_invoice_id',
                        'edit_customer_id', 
                        'edit_issue_date',
                        'edit_due_date'
                    ];
                    
                    for (const elementId of requiredElements) {
                        const element = document.getElementById(elementId);
                        if (!element) {
                            throw new Error(`Required element not found: ${elementId}`);
                        }
                        console.log(`${elementId}:`, element.value);
                    }
                
                // Collect items data
                const items = [];
                document.querySelectorAll('.edit-invoice-item').forEach(row => {
                    const productSelect = row.querySelector('.edit-product-select');
                    const productId = productSelect.value;
                    const productName = productSelect.options[productSelect.selectedIndex]?.text || '';
                    const quantity = parseFloat(row.querySelector('.edit-quantity').value) || 0;
                    const unitPrice = parseFloat(row.querySelector('.edit-unit-price').value) || 0;
                    
                    if (productId && quantity > 0 && unitPrice > 0) {
                        items.push({
                            product_id: productId,
                            product_name: productName,
                            quantity: quantity,
                            unit_price: unitPrice
                        });
                    }
                });
                
                if (items.length === 0) {
                    alert('Please add at least one item to the invoice.');
                    return;
                }
                
                const updateData = {
                    invoice_id: invoiceId,
                    customer_id: formData.get('customer_id'),
                    issue_date: formData.get('issue_date'),
                    due_date: formData.get('due_date'),
                    notes: formData.get('notes'),
                    items: items
                };
                
                console.log('Update data:', updateData);
                
                // Show loading state
                const submitBtn = this.querySelector('button[type="submit"]');
                const originalContent = submitBtn.innerHTML;
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
                
                // Send update request
                fetch('../api/update_invoice.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(updateData)
                })
                .then(response => {
                    console.log('Response status:', response.status);
                    console.log('Response headers:', response.headers);
                    
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    
                    return response.text().then(text => {
                        console.log('Raw response:', text);
                        try {
                            return JSON.parse(text);
                        } catch (e) {
                            console.error('JSON parse error:', e);
                            throw new Error('Invalid JSON response: ' + text);
                        }
                    });
                })
                .then(data => {
                    console.log('Parsed data:', data);
                    
                    if (data.success) {
                        alert('Invoice updated successfully!');
                        
                        // Close modal
                        const editModal = bootstrap.Modal.getInstance(document.getElementById('editInvoiceModal'));
                        editModal.hide();
                        
                        // Refresh the page to show updated data
                        location.reload();
                    } else {
                        let errorMsg = 'Error updating invoice: ' + (data.message || 'Unknown error');
                        if (data.debug) {
                            errorMsg += '\n\nDebug info: ' + JSON.stringify(data.debug, null, 2);
                        }
                        alert(errorMsg);
                    }
                })
                .catch(error => {
                    console.error('Error updating invoice:', error);
                    let errorMessage = 'Error updating invoice. Please try again.';
                    
                    // Try to get more detailed error information
                    if (error.message) {
                        errorMessage += '\n\nDetails: ' + error.message;
                    }
                    
                    // Check if it's a network error
                    if (error.name === 'TypeError' && error.message.includes('fetch')) {
                        errorMessage = 'Network error. Please check your connection and try again.';
                    }
                    
                    alert(errorMessage);
                })
                .finally(() => {
                    // Reset button state
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalContent;
                });
                
                } catch (error) {
                    console.error('Error in form submission:', error);
                    alert('Error preparing form data: ' + error.message);
                }
            });
            
            // Customer selection change in edit modal
            document.getElementById('edit_customer_id').addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const emailInput = document.getElementById('edit_customer_email');
                
                if (selectedOption.dataset.email) {
                    emailInput.value = selectedOption.dataset.email;
                } else {
                    emailInput.value = '';
                }
            });
            
            // Edit tax checkbox event listener
            const editTaxCheckbox = document.getElementById('edit_applyTaxCheckbox');
            if (editTaxCheckbox) {
                editTaxCheckbox.addEventListener('change', calculateEditTotals);
            }
        });
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 